-- create workload stored procedures for Microsoft SQL Server

-- random inputs: 
--   @randbase 0..4,999,999
--   @randtwomill 0..1,999,999
--   @randfivemill 0..4,999,999
--   @randfivemill1, randfivemill2, randfivemill3, randfivemill4, randfivemill5 0..4,999,999 
--   @randbillion 0..999999999

----------

-- procedure oltp_read_select_01 (randfivemill)
-- single random row select from the largest hundred-type table we have
-- uses: fivemill
-- needs index: fivemill.h_key
-- returns 1 row

create proc oltp_read_select_01 @randfivemill int as
	select	h_key, h_code, h_date, h_signed, h_name
	from	fivemill
	where	h_key = @randfivemill
go

-- procedure oltp_read_select_02 (randfivemill)
-- single table range select on an int, order by on a string
-- uses: sevmill
-- needs index: sevmill.u_key, sevmill.u_name
-- returns 201 rows

create proc oltp_read_select_02 @randfivemill int as
	select	h_key, h_code, h_date, h_signed, h_name
	from	hundred
	where	hundred.h_key between @randfivemill and @randfivemill+200
	order by hundred.h_name
go

--

-- procedure oltp_read_select_03
-- select 1 fixed row on an indexed int field
-- uses: tenpct
-- needs index: tenpct.t_key

create proc oltp_read_select_03 as
	select t_key, t_int, t_signed, t_double, t_code, t_name
	from tenpct
	where t_key = 1001
go

-- procedure oltp_read_select_04
-- select 1 fixed row on an indexed text field (non-clustered index)
-- uses: updates
-- needs index: updates.p_code
-- ~5.4 seconds

create proc oltp_read_select_04 as
	select p_key, p_int, p_signed, p_double, p_code, p_name
	from updates
	where p_code = 'BENCHMARKS'
go

-- procedure oltp_read_select_05
-- select 100 rows
-- uses: updates
-- needs index: updates.p_key

create proc oltp_read_select_05 as
	select p_key, p_int, p_signed, p_code, p_double, p_name
	from updates
	where p_key <= 100
go

-- procedure oltp_read_select_06
-- select 50 rows
-- uses: updates
-- needs index: updates.p_int

create proc oltp_read_select_06 @randbase int as
	select p_key, p_int, p_signed, p_code, p_double, p_name
	from updates
	where (p_int >= @randbase) and (p_int < @randbase+50)
go

-- procedure oltp_read_select_07
-- select 5 rows
-- uses: uniques
-- needs index: uniques.u_key

create proc oltp_read_select_07 as
	SELECT u_key, u_int, u_signed, u_code, u_double, u_name
	FROM uniques
	WHERE u_key <= 1000
go

-- procedure oltp_read_select_08 (randbillion)
-- uses: tenpct
-- needs index: tenpct.t_name
-- returns ~8 rows (depends on random input)

create proc oltp_read_select_08 @randbillion int as
	SELECT t_key, t_int, t_signed, t_code, t_double, t_name
	FROM tenpct
	WHERE t_name = 'THE+ASAP+BENCHMARKS+' and
	((t_key > @randbillion) and (t_key < @randbillion+10000))
go

-- procedure oltp_read_select_09
-- uses: tenpct
-- needs index: tenpct.t_signed
-- select 1 row (minimum value)

create proc oltp_read_select_09 as
	SELECT t_key, t_int, t_signed, t_code, t_double, t_name
	FROM tenpct
	WHERE t_signed <= -500000000
go

-- procedure oltp_read_select_10
-- select 2.5% of the rows (250 rows)
-- uses: tenthou
-- needs index: tenthou.t_signed

create proc oltp_read_select_10 as
	select t_key, t_int, t_signed, t_code, t_double, t_name
	from tenthou
	where t_signed <= -475000000
	order by t_key
go

-- procedure oltp_read_select_11
-- tablescan (row doesn't exist) on an unindexed field
-- uses: twomill
-- needs index: twomill.p_int
-- ~2.8 seconds

create proc oltp_read_select_11 as
	select *
	from twomill
	where p_int = 1
go

-- procedure oltp_read_select_12
-- select 0.01% of rows (479 rows)
-- uses: tenpct
-- needs index: tenpct.t_name, tenpct.t_signed
-- ~3 seconds

create proc oltp_read_select_12 as
	select t_key, t_int, t_signed, t_code, t_double, t_name
	from tenpct
	where t_name like 'THE+%' and t_signed > 499000000
go

-- procedure oltp_read_select_13
-- select 0.05% of rows (2500 rows, actually 2191 rows)
-- uses: tenpct
-- needs index: tenpct.t_date
-- long

create proc oltp_read_select_13 as
	select t_key, t_int, t_signed, t_code, t_double, t_name
	from tenpct
	where t_date > '12/15/1999'
	order by t_date
go

-- procedure oltp_read_select_14
-- select small number of rows (1642 rows)
-- uses: tenpct
-- needs index: tenpct.t_date

create proc oltp_read_select_14 as
	select t_key, t_int, t_signed, t_code, t_double, t_name
	from tenpct
	where t_date between '01/10/1942' and '01/21/1942'
go

-- procedure oltp_read_select_15
-- tablescan on an indexed character field (row doesn't exist)
-- uses: uniques
-- needs index: uniques.u_code

create proc oltp_read_select_15 as
	select *
	from uniques
	where u_code = 'xxxxxxxxxx'
go

-- procedure oltp_read_select_16
-- select 10,000 rows
-- uses: hunthou
-- needs index: hunthou.h_address, hunthou.h_signed
-- ~7.6 seconds

create proc oltp_read_select_16 as
	select distinct hunthou.h_address, hunthou.h_signed
	from hunthou
go

-- procedure oltp_read_select_17
-- select 10% of the rows in the table (1000 rows)
-- uses: tenthou
-- needs index: tenthou.t_decim

create proc oltp_read_select_17 as
	select distinct t_decim
	from tenthou
go

-- procedure oltp_read_select_18
-- select 1% of the table (100 rows)
-- uses: tenthou
-- needs index: tenthou.t_signed, tenthou.t_name

create proc oltp_read_select_18 as
	select t_name
	from tenthou
	where t_signed <= -490000000
	order by t_name
go

--

-- procedure oltp_read_join_01 (randbase)
-- single random row select, two table low cardinality join on an int
-- uses: updates, hundred
-- needs index: updates.p_key, hundred.h_key

create proc oltp_read_join_01 @randbase int as
	select	updates.p_key, updates.p_code, 
		hundred.h_date, hundred.h_signed, hundred.h_name, hundred.h_address
	from	updates, hundred
	where	updates.p_key = @randbase and
		updates.p_key = hundred.h_key
go

-- procedure oltp_read_join_02 (randtwomill)
-- single table range select on an int, two table low cardinality join on a string, order by on a date
-- uses: uniques, twomill
-- needs index: uniques.u_code (x2), twomill.p_key, twomill.p_code, twomill.p_date
-- returns ~1 row

create proc oltp_read_join_02 @randtwomill int as
	select	twomill.p_key, twomill.p_code, 
		uniques.u_date, uniques.u_signed, uniques.u_name
	from	twomill, uniques
	where	twomill.p_key between @randtwomill and (@randtwomill+50)
			and twomill.p_code = uniques.u_code
	order by twomill.p_date, uniques.u_code
go

-- procedure oltp_read_join_03 (randfivemill1, randfivemill2, randfivemill3, randfivemill4, randfivemill5)
-- single table in (group) select, two table low cardinality join, order by on a string
-- uses: uniques, fivemill
-- needs index: uniques.u_code, fivemill.h_key, fivemill.h_code (x2)
-- call using oltp_read_join_03 1,2,3,4,5
-- returns 5 rows

create proc oltp_read_join_03 @randfivemill1 int, @randfivemill2 int, @randfivemill3 int, @randfivemill4 int, @randfivemill5 int as
	select	fivemill.h_key, fivemill.h_code,
		uniques.u_date, uniques.u_signed, uniques.u_name
	from	fivemill, uniques
	where	fivemill.h_key in (@randfivemill1,@randfivemill2,@randfivemill3,@randfivemill4,@randfivemill5) 
	and	fivemill.h_code = uniques.u_code
	order by fivemill.h_code
go

-- procedure oltp_read_join_04 (randbase)
-- single random row select, two table low cardinality join on an int
-- uses: updates, hundred
-- needs index: updates.p_key, updates.p_code, hundred.h_key, hundred.h_date
-- deliberately out-of-order where clause
-- returns 49 rows

create proc oltp_read_join_04 @randbase int as
	select	updates.p_key, updates.p_code, 
		hundred.h_date, hundred.h_signed, hundred.h_name, hundred.h_address
	from	updates, hundred
	where	((hundred.h_key > @randbase) and (hundred.h_key < (@randbase+50))) and
		updates.p_key = hundred.h_key
	order by updates.p_code, hundred.h_date
go

-- procedure oltp_read_join_05
-- returns 51 rows
-- uses: uniques, tenpct
-- needs index: uniques.u_key, tenpct.t_key

create proc oltp_read_join_05 as
	select uniques.u_key, uniques.u_name, tenpct.t_key, tenpct.t_signed
	from uniques, tenpct
	where uniques.u_key >= 1000000000
		and tenpct.t_int <= uniques.u_int
		and tenpct.t_key >= 999990000
go

-- procedure oltp_read_join_06
-- returns 2 rows
-- uses: uniques, sevmill
-- needs index: uniques.u_key, uniques.u_code, sevmill.u_key

create proc oltp_read_join_06 as
	select distinct uniques.u_signed, uniques.u_name, sevmill.u_signed, sevmill.u_name
	from uniques, sevmill
	where uniques.u_key = sevmill.u_key and
		uniques.u_code like 'AAA%'
go

-- procedure oltp_read_join_07
-- returns 1 row
-- uses: uniques, hundred
-- needs index: uniques.u_key, hundred.h_key

create proc oltp_read_join_07 as
	select uniques.u_signed, uniques.u_name, hundred.h_signed, hundred.h_name
	from uniques, hundred
	where uniques.u_key = hundred.h_key
	and uniques.u_key = 1001
go

-- procedure oltp_read_join_08
-- returns 1 row
-- uses: uniques, hundred
-- needs index: uniques.u_code, hundred.h_code

create proc oltp_read_join_08 as
	select uniques.u_signed, uniques.u_name, hundred.h_signed, hundred.h_name
	from uniques, hundred
	where uniques.u_code = hundred.h_code
	and uniques.u_code = 'BENCHMARKS'
go

-- procedure oltp_read_join_09
-- returns 1 row
-- uses: uniques, hundred, tenpct
-- needs index: uniques.u_key, hundred.h_key, tenpct.t_key

create proc oltp_read_join_09 as
	select uniques.u_signed, uniques.u_date, hundred.h_signed, hundred.h_date, tenpct.t_signed, tenpct.t_date
	from uniques, hundred, tenpct
	where uniques.u_key = hundred.h_key
	and uniques.u_key = tenpct.t_key
	and uniques.u_key = 1001
go

-- procedure oltp_read_join_10
-- returns 1 row
-- uses: uniques, hundred, tenpct
-- needs index: uniques.u_code, hundred.h_code, tenpct.t_code

create proc oltp_read_join_10 as
	select uniques.u_signed, uniques.u_date, hundred.h_signed, hundred.h_date, tenpct.t_signed, tenpct.t_date
	from uniques, hundred, tenpct
	where uniques.u_code = hundred.h_code
	and uniques.u_code = tenpct.t_code
	and uniques.u_code = 'BENCHMARKS'
go

-- procedure oltp_read_join_11
-- returns 1 row
-- uses: uniques, hundred, tenpct, updates
-- needs index: uniques.u_key, hundred.h_key, tenpct.t_key, updates.p_key

create proc oltp_read_join_11 as
	select uniques.u_date, hundred.h_date, tenpct.t_date, updates.p_date
	from uniques, hundred, tenpct, updates
	where uniques.u_key = hundred.h_key
	and uniques.u_key = tenpct.t_key
	and uniques.u_key = updates.p_key
	and uniques.u_key = 1001
go

-- procedure oltp_read_join_12
-- returns 1 row
-- uses: uniques, hundred, tenpct, updates
-- needs index: uniques.u_code, hundred.h_code, tenpct.t_code, updates.p_code
-- ~5.4 seconds

create proc oltp_read_join_12 as
	select uniques.u_date, hundred.h_date, tenpct.t_date, updates.p_date
	from uniques, hundred, tenpct, updates
	where uniques.u_code = hundred.h_code
	and uniques.u_code = tenpct.t_code
	and uniques.u_code = updates.p_code
	and uniques.u_code = 'BENCHMARKS'
go

-- procedure oltp_read_join_13
-- 4911 rows returned
-- uses: uniques, tenpct
-- needs index: uniques.u_key, uniques.u_signed, tenpct.t_key, tenpct.t_signed
-- ~1.1 seconds

create proc oltp_read_join_13 as
	SELECT uniques.u_name, uniques.u_int
	FROM uniques
	Where uniques.u_key <= 2000000
	and uniques.u_signed >
		(select avg(tenpct.t_signed)
		from tenpct
		where uniques.u_key = tenpct.t_key)
go

-- procedure oltp_read_join_14
-- returns 10 rows
-- uses: tenpct, tenpct
-- needs index: tenpct.h_signed, tenpct.h_name, tenpct.t_name

create proc oltp_read_join_14 as
	select tenthou.t_key, tenthou.t_name, tenthou.t_signed, tenthou.t_code, ISNULL(tenpct.t_key, 0)
	from tenthou, tenpct
	where tenthou.t_code *= tenpct.t_code
	and tenthou.t_key <= 1000000
go

------
------

-- procedure oltp_write_update_1 (randbase)
-- single table range select and update on an int
-- needs index: updates.p_key
-- locks: updates.p_int
-- no returned data
-- competes with oltp_read_select_06 and oltp_read_join_11
-- no permanent changes to data set

create procedure oltp_write_update_1 @randbase int as

set transaction isolation level repeatable read
	-- increment value
	begin transaction
		update	updates
			set	updates.p_int = updates.p_int+1
			where	updates.p_key between @randbase and @randbase+200
		if (@@error <> 0) goto on_error
	commit transaction
	-- decrement value
	begin transaction
		update	updates
			set	updates.p_int = updates.p_int-1
			where	updates.p_key between @randbase and @randbase+200
		if (@@error <> 0) goto on_error
	commit transaction
	return (0)
on_error:
	rollback transaction
	return (1)
go

--

-- procedure oltp_write_update_2 (twomill1, twomill2, twomill3, twomill4, twomill5)
-- single table in (group) select and update on an int
-- needs index: twomill.p_key
-- locks: twomill.p_key
-- competes with oltp_read_join_02
-- no returned data
-- no permanent changes to data set

create procedure oltp_write_update_2 @randfivemill1 int, @randfivemill2 int, @randfivemill3 int, @randfivemill4 int, @randfivemill5 int as

set transaction isolation level repeatable read
	-- increment value
	begin transaction
		update	twomill
			set	twomill.p_key = twomill.p_key+2000000
				where	twomill.p_key in (@randfivemill1, @randfivemill2, @randfivemill3, @randfivemill4, @randfivemill5)
		if (@@error <> 0) goto on_error
	commit transaction
	-- decrement value
	begin transaction
		update	twomill
			set	twomill.p_key = twomill.p_key-2000000
				where	twomill.p_key in (@randfivemill1, @randfivemill2, @randfivemill3, @randfivemill4, @randfivemill5)
		if (@@error <> 0) goto on_error
	commit transaction
	return (0)
on_error:
	rollback transaction
	return (1)
go

--

-- procedure oltp_write_insert_1 (randfivemill)
-- single row select and insert, single row delete
-- needs index: fivemill.h_key
-- changes: fivemill, del_history
-- competes with oltp_read_join_03

create procedure oltp_write_insert_1 @randfivemill int as

set transaction isolation level repeatable read
	-- copy the records to a safe place
	begin transaction
		-- find records and copy
		insert	del_history 
			select * from fivemill where fivemill.h_key = @randfivemill
		if (@@error <> 0) goto on_error
		-- delete the records
		delete from fivemill where fivemill.h_key = @randfivemill
		if (@@error <> 0) goto on_error
	commit transaction
	-- copy the records back again
	begin transaction
		-- find records and copy
		insert	fivemill
			select * from del_history where del_history.h_key = @randfivemill
		if (@@error <> 0) goto on_error
		-- delete the records
		delete from del_history where del_history.h_key = @randfivemill
		if (@@error <> 0) goto on_error
	commit transaction
	return (0)
on_error:
	rollback transaction
	return (1)
go

--

-- procedure oltp_write_insert_2 (randbase)
--
-- needs index: updates.p_key, updates.p_code
-- changes: updates.p_int, updates.p_code

create procedure oltp_write_insert_2 @randbase int as

set transaction isolation level repeatable read

declare
	@textstr	char(12),
	@int_val	int,
	@signed_val	int

	begin transaction

		--  save old values
		select	@signed_val = updates.p_signed, @int_val = updates.p_int
		from	updates
			where	updates.p_key = @randbase and 
				updates.p_code != 'Updated'
		if @@rowcount = 0 begin
			rollback transaction
			raiserror  20405 'oltp_write_insert_2: no matching record found in updates (possibly already modified).'
			return (1)
		end

		-- set value of textstr
		if (@signed_val < 0) 
			select @textstr = 'low value'
		else 
			select @textstr = 'high value'

		-- update updates and set updates.p_code to special flag
		update	updates
			set updates.p_signed = updates.p_signed+1, updates.p_code = 'Updated'
			where	updates.p_key = @randbase and
				updates.p_code != 'Updated'
		if @@rowcount = 0 begin
			rollback transaction
			raiserror  20405 'oltp_write_insert_2: couldn''t update found record.'
			return (1)
		end

		-- insert log of change into upd_history
		insert into upd_history 
			values (@randbase, @int_val, @signed_val, getdate(), @textstr)

		-- check for correct operation:
		-- select upd_history.x_key, updates.p_key, upd_history.x_signed, updates.p_signed
		--   from upd_history, updates
		--   where upd_history.x_key = updates.p_key and
		--	upd_history.x_signed != updates.p_signed-1
		-- should be an empty set

	commit transaction
	return (0)
go

--

